<?php
/**
 * Public Posts Routes
 */

function getPosts() {
    $db = Database::getInstance()->getConnection();
    
    $page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
    $limit = POSTS_PER_PAGE;
    $offset = ($page - 1) * $limit;
    
    $category = $_GET['category'] ?? null;
    $search = $_GET['search'] ?? null;
    
    // Build query
    $where = ["p.status = 'published'"];
    $params = [];
    
    if ($category) {
        $where[] = "c.slug = :category";
        $params[':category'] = $category;
    }
    
    if ($search) {
        $where[] = "(p.title LIKE :search OR p.content LIKE :search OR p.excerpt LIKE :search)";
        $params[':search'] = "%$search%";
    }
    
    $whereClause = implode(' AND ', $where);
    
    // Get total count
    $countQuery = "SELECT COUNT(*) as total FROM posts p 
                   LEFT JOIN categories c ON p.category_id = c.id 
                   WHERE $whereClause";
    $countStmt = $db->prepare($countQuery);
    $countStmt->execute($params);
    $total = $countStmt->fetch()['total'];
    
    // Get posts
    $query = "SELECT 
                p.id, p.title, p.slug, p.excerpt, p.featured_image,
                p.created_at, p.published_at, p.views,
                c.name as category_name, c.slug as category_slug,
                u.name as author_name
              FROM posts p
              LEFT JOIN categories c ON p.category_id = c.id
              LEFT JOIN users u ON p.author_id = u.id
              WHERE $whereClause
              ORDER BY p.published_at DESC
              LIMIT :limit OFFSET :offset";
    
    $stmt = $db->prepare($query);
    foreach ($params as $key => $value) {
        $stmt->bindValue($key, $value);
    }
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    
    $posts = $stmt->fetchAll();
    
    echo json_encode([
        'success' => true,
        'data' => [
            'posts' => $posts,
            'pagination' => [
                'page' => $page,
                'limit' => $limit,
                'total' => (int)$total,
                'pages' => ceil($total / $limit)
            ]
        ]
    ]);
}

function getPostBySlug($slug) {
    $db = Database::getInstance()->getConnection();
    
    $query = "SELECT 
                p.*,
                c.name as category_name, c.slug as category_slug,
                u.name as author_name
              FROM posts p
              LEFT JOIN categories c ON p.category_id = c.id
              LEFT JOIN users u ON p.author_id = u.id
              WHERE p.slug = :slug AND p.status = 'published'";
    
    $stmt = $db->prepare($query);
    $stmt->execute([':slug' => $slug]);
    $post = $stmt->fetch();
    
    if (!$post) {
        http_response_code(404);
        echo json_encode([
            'success' => false,
            'message' => 'Post not found'
        ]);
        return;
    }
    
    // Increment view count
    $updateViews = $db->prepare("UPDATE posts SET views = views + 1 WHERE id = :id");
    $updateViews->execute([':id' => $post['id']]);
    
    // Get tags
    $tagsQuery = "SELECT t.name, t.slug FROM tags t
                  INNER JOIN post_tags pt ON t.id = pt.tag_id
                  WHERE pt.post_id = :post_id";
    $tagsStmt = $db->prepare($tagsQuery);
    $tagsStmt->execute([':post_id' => $post['id']]);
    $post['tags'] = $tagsStmt->fetchAll();
    
    // Get related posts
    $relatedQuery = "SELECT id, title, slug, featured_image, excerpt
                     FROM posts
                     WHERE category_id = :category_id 
                     AND id != :post_id 
                     AND status = 'published'
                     ORDER BY published_at DESC
                     LIMIT 3";
    $relatedStmt = $db->prepare($relatedQuery);
    $relatedStmt->execute([
        ':category_id' => $post['category_id'],
        ':post_id' => $post['id']
    ]);
    $post['related_posts'] = $relatedStmt->fetchAll();
    
    echo json_encode([
        'success' => true,
        'data' => $post
    ]);
}