<?php
/**
 * Blog API - Main Entry Point
 * Subdomain: blog.nosyradigital.com.ng
 */

header('Content-Type: application/json');

// CORS Configuration
$allowed_origins = [
    'https://nosyradigital.vercel.app',
    'https://www.nosyradigital.com.ng',
    'http://localhost:3000',
    'http://localhost:5173',
    'https://blog.nosyradigital.com.ng/'
];

$origin = $_SERVER['HTTP_ORIGIN'] ?? '';

if (in_array($origin, $allowed_origins)) {
    header("Access-Control-Allow-Origin: $origin");
} else {
    header("Access-Control-Allow-Origin: https://www.nosyradigital.com.ng");
}

header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');
header('Access-Control-Allow-Credentials: true');
header('Access-Control-Max-Age: 86400');

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Error handling
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/logs/php-errors.log');

// Autoloader
spl_autoload_register(function ($class) {
    $file = __DIR__ . '/classes/' . str_replace('\\', '/', $class) . '.php';
    if (file_exists($file)) {
        require_once $file;
    }
});

// Load configuration
require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/config/config.php';

// Simple Router
$request_uri = $_SERVER['REQUEST_URI'];
$request_method = $_SERVER['REQUEST_METHOD'];

// Remove query string and base path
$path = parse_url($request_uri, PHP_URL_PATH);
$path = str_replace('/api', '', $path);
$path = trim($path, '/');

// Route handling
try {
    // Public routes
    if (preg_match('/^posts$/', $path) && $request_method === 'GET') {
        require_once __DIR__ . '/routes/posts.php';
        getPosts();
    }
    elseif (preg_match('/^posts\/([a-z0-9-]+)$/', $path, $matches) && $request_method === 'GET') {
        require_once __DIR__ . '/routes/posts.php';
        getPostBySlug($matches[1]);
    }
    elseif (preg_match('/^categories$/', $path) && $request_method === 'GET') {
        require_once __DIR__ . '/routes/categories.php';
        getCategories();
    }
    elseif (preg_match('/^categories\/([a-z0-9-]+)\/posts$/', $path, $matches) && $request_method === 'GET') {
        require_once __DIR__ . '/routes/categories.php';
        getCategoryPosts($matches[1]);
    }
    
    // Admin routes (require authentication)
    elseif (preg_match('/^admin\/login$/', $path) && $request_method === 'POST') {
        require_once __DIR__ . '/routes/auth.php';
        login();
    }
    elseif (preg_match('/^admin\/posts$/', $path) && $request_method === 'POST') {
        require_once __DIR__ . '/routes/admin/posts.php';
        createPost();
    }
    elseif (preg_match('/^admin\/posts\/(\d+)$/', $path, $matches) && $request_method === 'PUT') {
        require_once __DIR__ . '/routes/admin/posts.php';
        updatePost($matches[1]);
    }
    elseif (preg_match('/^admin\/posts\/(\d+)$/', $path, $matches) && $request_method === 'DELETE') {
        require_once __DIR__ . '/routes/admin/posts.php';
        deletePost($matches[1]);
    }
    elseif (preg_match('/^admin\/upload$/', $path) && $request_method === 'POST') {
        require_once __DIR__ . '/routes/admin/upload.php';
        uploadImage();
    }
    elseif (preg_match('/^admin\/categories$/', $path) && $request_method === 'POST') {
        require_once __DIR__ . '/routes/admin/categories.php';
        createCategory();
    }
    
    // 404 Not Found
    else {
        http_response_code(404);
        echo json_encode([
            'success' => false,
            'message' => 'Endpoint not found',
            'path' => $path
        ]);
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Internal server error',
        'error' => $e->getMessage()
    ]);
}